// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "extensions/browser/api/declarative_webrequest/webrequest_condition.h"

#include <memory>
#include <set>

#include "base/test/values_test_util.h"
#include "base/values.h"
#include "components/url_matcher/url_matcher_constants.h"
#include "extensions/browser/api/declarative_webrequest/webrequest_constants.h"
#include "extensions/browser/api/web_request/web_request_info.h"
#include "testing/gtest/include/gtest/gtest.h"

using url_matcher::URLMatcher;
using url_matcher::URLMatcherConditionSet;

namespace extensions {

TEST(WebRequestConditionTest, CreateCondition) {
  URLMatcher matcher;

  std::string error;
  std::unique_ptr<WebRequestCondition> result;

  // Test wrong condition name passed.
  error.clear();
  constexpr const char kWrongNameCondition[] = R"({
    "invalid": "foobar",
    "instanceType": "declarativeWebRequest.RequestMatcher",
  })";
  result = WebRequestCondition::Create(
      nullptr, matcher.condition_factory(),
      base::test::ParseJson(kWrongNameCondition), &error);
  EXPECT_FALSE(error.empty());
  EXPECT_FALSE(result.get());

  // Test wrong datatype in host_suffix.
  constexpr const char kWrongDataTypeCondition[] = R"({
    "url": [],
    "instanceType": "declarativeWebRequest.RequestMatcher",
  })";
  error.clear();
  result = WebRequestCondition::Create(
      nullptr, matcher.condition_factory(),
      base::test::ParseJson(kWrongDataTypeCondition), &error);
  EXPECT_FALSE(error.empty());
  EXPECT_FALSE(result.get());

  // Test success (can we support multiple criteria?)
  error.clear();
  constexpr const char kMultipleCriteriaCondition[] = R"({
    "resourceType": ["main_frame"],
    "url": { "hostSuffix": "example.com" },
    "instanceType": "declarativeWebRequest.RequestMatcher",
  })";
  result = WebRequestCondition::Create(
      nullptr, matcher.condition_factory(),
      base::test::ParseJson(kMultipleCriteriaCondition), &error);
  EXPECT_EQ("", error);
  ASSERT_TRUE(result.get());

  URLMatcherConditionSet::Vector url_matcher_condition_set;
  result->GetURLMatcherConditionSets(&url_matcher_condition_set);
  matcher.AddConditionSets(url_matcher_condition_set);

  const GURL http_url("http://www.example.com");
  WebRequestInfoInitParams match_params;
  match_params.url = http_url;
  match_params.type = blink::mojom::ResourceType::kMainFrame;
  match_params.web_request_type = WebRequestResourceType::MAIN_FRAME;
  WebRequestInfo match_request_info(std::move(match_params));
  WebRequestData data(&match_request_info, ON_BEFORE_REQUEST);
  WebRequestDataWithMatchIds request_data(&data);
  request_data.url_match_ids = matcher.MatchURL(http_url);
  EXPECT_EQ(1u, request_data.url_match_ids.size());
  EXPECT_TRUE(result->IsFulfilled(request_data));

  const GURL https_url("https://www.example.com");
  WebRequestInfoInitParams wrong_resource_type_params;
  wrong_resource_type_params.url = https_url;
  wrong_resource_type_params.type = blink::mojom::ResourceType::kSubFrame;
  wrong_resource_type_params.web_request_type =
      WebRequestResourceType::SUB_FRAME;
  WebRequestInfo wrong_resource_type_request_info(
      std::move(wrong_resource_type_params));
  data.request = &wrong_resource_type_request_info;
  request_data.url_match_ids = matcher.MatchURL(http_url);
  // Make sure IsFulfilled does not fail because of URL matching.
  EXPECT_EQ(1u, request_data.url_match_ids.size());
  EXPECT_FALSE(result->IsFulfilled(request_data));
}

TEST(WebRequestConditionTest, IgnoreConditionFirstPartyForCookies) {
  // firstPartyForCookiesUrl is deprecated, but must still be accepted in
  // parsing.
  URLMatcher matcher;

  std::string error;
  std::unique_ptr<WebRequestCondition> result;

  constexpr const char kCondition[] = R"({
    "firstPartyForCookiesUrl": { "hostPrefix": "fpfc"},
    "instanceType": "declarativeWebRequest.RequestMatcher",
  })";
  result =
      WebRequestCondition::Create(nullptr, matcher.condition_factory(),
                                  base::test::ParseJson(kCondition), &error);
  EXPECT_EQ("", error);
  ASSERT_TRUE(result.get());
}

TEST(WebRequestConditionTest, IgnoreConditionThirdPartForCookies) {
  // thirdPartyForCookies is deprecated, but must still be accepted in
  // parsing.
  URLMatcher matcher;

  std::string error;
  std::unique_ptr<WebRequestCondition> result;

  constexpr const char kConditionTrue[] = R"({
    "thirdPartyForCookies": true,
    "instanceType": "declarativeWebRequest.RequestMatcher",
  })";
  result = WebRequestCondition::Create(nullptr, matcher.condition_factory(),
                                       base::test::ParseJson(kConditionTrue),
                                       &error);
  EXPECT_EQ("", error);
  ASSERT_TRUE(result.get());

  constexpr const char kConditionFalse[] = R"({
    "thirdPartyForCookies": false,
    "instanceType": "declarativeWebRequest.RequestMatcher",
  })";
  result = WebRequestCondition::Create(nullptr, matcher.condition_factory(),
                                       base::test::ParseJson(kConditionFalse),
                                       &error);
  EXPECT_EQ("", error);
  ASSERT_TRUE(result.get());
}

// Conditions without UrlFilter attributes need to be independent of URL
// matching results. We test here that:
//   1. A non-empty condition without UrlFilter attributes is fulfilled iff its
//      attributes are fulfilled.
//   2. An empty condition (in particular, without UrlFilter attributes) is
//      always fulfilled.
TEST(WebReq