/*
  Copyright (c) 2010-2025, Intel Corporation

  SPDX-License-Identifier: BSD-3-Clause
*/

/** @file type.h
    @brief File with declarations for classes related to type representation
*/

#pragma once

#include "ispc.h"
#include "util.h"

#include <llvm/ADT/SmallVector.h>
#include <llvm/IR/DerivedTypes.h>
#include <llvm/IR/Type.h>

namespace ispc {

class ConstExpr;
class StructType;

/** Types may have uniform, varying, SOA, or unbound variability; this
    struct is used by Type implementations to record their variability.
*/
struct Variability {
    enum VarType { Unbound, Uniform, Varying, SOA };

    Variability(VarType t = Unbound, int w = 0) : type(t), soaWidth(w) {}

    bool operator==(const Variability &v) const { return v.type == type && v.soaWidth == soaWidth; }
    bool operator!=(const Variability &v) const { return v.type != type || v.soaWidth != soaWidth; }

    bool operator==(const VarType &t) const { return type == t; }
    bool operator!=(const VarType &t) const { return type != t; }

    std::string GetString() const;
    std::string MangleString() const;

    VarType type;
    int soaWidth;
};

/** Enumerant that records each of the types that inherit from the Type
    baseclass. */
enum TypeId {
    ATOMIC_TYPE,             // 0
    ENUM_TYPE,               // 1
    POINTER_TYPE,            // 2
    ARRAY_TYPE,              // 3
    VECTOR_TYPE,             // 4
    STRUCT_TYPE,             // 5
    UNDEFINED_STRUCT_TYPE,   // 6
    REFERENCE_TYPE,          // 7
    FUNCTION_TYPE,           // 8
    TEMPLATE_TYPE_PARM_TYPE, // 9
};

/** Enumerant that records syntax types that can be generated by Type::GetDeclaration. */
enum DeclarationSyntax { C, CPP };

/** @brief Interface class that defines the type abstraction.

    Abstract base class that defines the interface that must be implemented
    for all types in the language.
 */
class Type : public Traceable {
  public:
    /** Returns true if the underlying type is boolean.  In other words,
        this is true for individual bools and for short-vectors with
        underlying bool type, but not for arrays of bools. */
    virtual bool IsBoolType() const { return false; }

    /** Returns true if the underlying type is float or double.  In other
        words, this is true for individual floats/doubles and for
        short-vectors of them, but not for arrays of them. */
    virtual bool IsFloatType() const { return false; }

    /** Returns true if the underlying type is an integer type.  In other
        words, this is true for individual integers and for short-vectors
        of integer types, but not for arrays of integer types. */
    virtual bool IsIntType() const { return false; }

    /** Returns true if the underlying type is unsigned.  In other words,
        this is true for unsigned integers and short vectors of unsigned
        integer types. */
    virtual bool IsUnsignedType() const { return false; }

    /** Returns true if the underlying type is signed.  In other words,
        this is true for signed integers and short vectors of signed
        integer types. */
    virtual bool IsSignedType() const { return false; }

    /** Returns true if the underlying type is either a pointer type */
    bool IsPointerType() const;

    /** Returns true if the underlying type is a array type */
    bool IsArrayType() const;

    /** Returns true if the underlying type is an atomic type */
    bool IsAtomicType() const;

    /** Returns true if the underlying type is an varying atomic or uniform vector type */
    bool IsVaryingAtomicOrUniformVectorType() const;

    /** Returns true if the underlying type is a varying atomic type */
    bool IsVaryingAtomic() const;

    /** Returns true if the underlying type is an uniform vector type */
    bool IsUniformVector() const;

    /** Returns true if the underlying type is a reference type */
    bool IsReferenceType() const;

    /** Returns true if the underlying type is vector type */
    bool IsVectorType() const;

    /** Returns true if the underlying type is either a pointer or an array */
    bool IsVoidType() const;

    /** Returns true if this type is 'const'-qualified. */
    bool IsConstType() const { return isConst; }

    /** Returns true if this type is complete. This is used to check for
        incomplete types (e.g. forward-declared structs) that are not
        allowed in certain contexts, e.g., when we need to allocate memory for
        them. */
    virtual bool IsCompleteType() const { return true; }

    /** Returns true if the underlying type is a float or integer type. */
    bool IsNumericType() const { return IsFloatType() || IsIntType(); }

    /** Returns true if the type is any kind of dependent type. */
    bool IsDependent() const;

    /** This function checks whether the size of the collection (e.g., an array or vector)
        is influenced by a template parameter, indicating that the count may vary based
        on the specific instantiation of the template. If the count is fixed and known
        at compile time, this function will return false.
        Returns true if the element count is dependent on a template parameter;
   */
    bool IsCountDependent() const;

    /** This function determines whether the type's definition relies on a template
        type parameter, indicating that the type may vary based on the specific
        instantiation of the template. If the type does not depend on
        template parameters, this function will return false.
        Returns true if the type is type dependent (based on template type parameter).
    */
    bool IsTypeDependent() const;

    /** Returns the variability of the type. */
    Variability GetVariability() const { return variability; }

    /** Returns the alignment of the type in bytes. */
    unsigned int GetAlignment() const { return alignment; }

    /** Returns true if the underlying type is uniform */
    bool IsUniformType() const { return GetVariability() == Variability::Uniform; }

    /** Returns true if the underlying type is varying */
    bool IsVaryingType() const { return GetVariability() == Variability::Varying; }

    /** Returns true if the type is laid out in "structure of arrays"
        layout. */
    bool IsSOAType() const { return GetVariability() == Variability::SOA; }

    /** Returns the structure of arrays width for SOA types.  This method
        returns zero for types with non-SOA variability. */
    int GetSOAWidth() const { return GetVariability().soaWidth; }

    /** Returns true if the underlying type's uniform/varying-ness is
        unbound. */
    bool HasUnboundVariability() const { return GetVariability() == Variability::Unbound; }

    /** Resolves dependent type with mapping of template types to concrete type passed as argeument.
        Do not handle variability resolution, which need to be done as a separate step.
     */
    virtual const Type *ResolveDependence(TemplateInstantiation &templInst) const { return this; }

    /** Resolves dependent type with mapping of template types to concrete type passed as argeument.
        Resolve variability of the type as well.
     */
    const Type *ResolveDependenceForTopType(TemplateInstantiation &templInst) const;

    /* Returns a type wherein any elements of the original type and
       contained types that have unbound variability have their variability
       set to the given variability. */
    virtual const Type *ResolveUnboundVariability(Variability v) const;

    /** Return a "uniform" instance of this type.  If the type is already
        uniform, its "this" pointer will be returned. */
    virtual const Type *GetAsUniformType() const;

    /** Return a "varying" instance of this type.  If the type is already
        varying, its "this" pointer will be returned. */
    virtual const Type *GetAsVaryingType() const;

    /** Get an instance of the type with unbound variability. */
    virtual const Type *GetAsUnboundVariabilityType() const;

    virtual const Type *GetAsSOAType(int width) const;

    /** If this is a signed integer type, return the unsigned version of
        the type.  Otherwise, return the original type. */
    virtual const Type *GetAsUnsignedType() const;

    /** If this is a signed integer type, return the unsigned version of
        the type.  Otherwise, return the original type. */
    virtual const Type *GetAsSignedType() const;

    /** Returns the basic root type of the given type.  For example, for an
        array or short-vector, this returns the element type.  For a struct
        or atomic type, it returns itself. */
    virtual const Type *GetBaseType() const { return this; };

    /** If this is a reference type, returns the type it is referring to.
        For all other types, just returns its own type. */
    virtual const Type *GetReferenceTarget() const;

    /** Get a const version of this type.  If it's already const, then the old
        Type pointer is returned. */
    virtual const Type *GetAsConstType() const;

    /** Get a non-const version of this type.  If it's already not const,
        then the old Type pointer is returned. */
    virtual const Type *GetAsNonConstType() const;

    /** Returns a new type that is the same as this type, but with the given
        alignment. */
    virtual const Type *GetAsAlignedType(unsigned int alignment) const;

    /** Returns a text representation of the type (for example, for use in
        warning and error messages). */
    virtual std::string GetString() const = 0;

    const SourcePos &GetSourcePos() const { return pos; }

    /** Returns a string that represents the mangled type (for use in
        mangling function symbol names for function overloading).  The
        various Types implementations of this method should collectively
        ensure that all of them use mangling schemes that are guaranteed
        not to clash. */
    virtual std::string Mangle() const = 0;

    /** Returns a string that is the declaration of the same type in C or C++
        syntax. The second argument controls which syntax is generated. */
    virtual std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const = 0;

    /** Returns the LLVM type corresponding to this ispc type. */
    virtual llvm::Type *LLVMType(llvm::LLVMContext *ctx) const = 0;

    /** Returns the LLVM storage type corresponding to this ispc type. */
    virtual llvm::Type *LLVMStorageType(llvm::LLVMContext *ctx) const;

    /** Returns the DIType (LLVM's debugging information structure),
        corresponding to this type. */
    virtual llvm::DIType *GetDIType(llvm::DIScope *scope) const = 0;

    /** Checks two types for equality.  Returns true if they are exactly
        the same, false otherwise. */
    static bool Equal(const Type *a, const Type *b);

    /** Checks two types for equality.  Returns true if they are exactly
        the same (ignoring const-ness of the type), false otherwise. */
    static bool EqualIgnoringConst(const Type *a, const Type *b);

    /** Given two types, returns the least general Type that is more general
        than both of them.  (i.e. that can represent their values without
        any loss of data.)  If there is no such Type, return nullptr.

        @param type0        First of the two types
        @param type1        Second of the two types
        @param pos          Source file position where the general type is
                            needed.
        @param reason       String describing the context of why the general
                            type is needed (e.g. "+ operator").
        @param forceVarying If \c true, then make sure that the returned
                            type is "varying".
        @param vecSize      The vector size of the returned type.  If non-zero,
                            the returned type will be a VectorType of the
                            more general type with given length.  If zero,
                            this parameter has no effect.
        @return             The more general type, based on the provided parameters.

        @todo the vecSize and forceVarying parts of this should probably be
        factored out and done separately in the cases when needed.

    */
    static const Type *MoreGeneralType(const Type *type0, const Type *type1, SourcePos pos, const char *reason,
                                       bool forceVarying = false, int vecSize = 0);

    /** Returns true if the given type is an atomic, enum, or pointer type
        (i.e. not an aggregation of multiple instances of a type or
        types.) */
    // TODO!: consider renaming to IsPrimitiveType
    static bool IsBasicType(const Type *type);

    /** Indicates which Type implementation this type is.  This value can
        be used to determine the actual type much more efficiently than
        using dynamic_cast. */
    const TypeId typeId;

  protected:
    // The mutable pointers (asOtherConstType, asUniformType, asVaryingType) are
    // initialized to nullptr instead of copying the pointers from 'other'
    // since these are cached values that will be recomputed when needed
    mutable const Type *asOtherConstType = {};
    mutable const Type *asUniformType = {};
    mutable const Type *asVaryingType = {};

    /** The variability of the type. */
    Variability variability = {};

    /** Indicates whether the type is const-qualified. */
    bool isConst = {};

    /** Source file position where the type is declared. */
    SourcePos pos = {};

    /** Alignment of the type in bytes. */
    mutable unsigned int alignment = 0;

    Type(TypeId id, Variability v = Variability(Variability::Unbound), bool is_const = false, SourcePos s = SourcePos(),
         unsigned int alignment = 0);

    /* These methods are useful to avoid calling the constructor directly.
     createWith methods are used to create a new type that is a shallow copy of
     the current object, but with some field changed. The value of these fields
     are changed after the object is created. */
    /** Create a copy of the current object without enumerating all fields. */
    virtual Type *create() const = 0;

    /** Create a copy then set new value of isConst member. */
    virtual const Type *createWithConst(bool newIsConst) const;

    /** Create a copy then set new value of variability member. */
    virtual const Type *createWithVariability(Variability newVariability) const;

    /** Clone method to allow cloning with new alignment */
    virtual const Type *createWithAlignment(unsigned int newAlignment) const;
};

/** @brief AtomicType represents basic types like floats, ints, etc.

    AtomicTypes can be either uniform or varying.  Unique instances of all
    of the possible <tt>AtomicType</tt>s are available in the static members
    like AtomicType::UniformInt32.  It is thus possible to compare
    AtomicTypes for equality with simple pointer equality tests; this is
    not true for the other Type implementations.
 */
class AtomicType : public Type {
  public:
    bool IsBoolType() const override;
    bool IsFloatType() const override;
    bool IsIntType() const override;
    bool IsUnsignedType() const override;
    bool IsSignedType() const override;

    const AtomicType *GetAsUnsignedType() const override;
    const AtomicType *GetAsSignedType() const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMStorageType(llvm::LLVMContext *ctx) const override;
    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    /** This enumerator records the basic types that AtomicTypes can be
        built from.  */
    // TODO!: consider renaming to something like TypeKind with prefix avoidin Base or Basic
    enum BasicType {
        TYPE_VOID,
        TYPE_BOOL,
        TYPE_INT1,
        TYPE_INT8,
        TYPE_UINT8,
        TYPE_INT16,
        TYPE_UINT16,
        TYPE_INT32,
        TYPE_UINT32,
        TYPE_FLOAT16,
        TYPE_FLOAT,
        TYPE_INT64,
        TYPE_UINT64,
        TYPE_DOUBLE,
        TYPE_DEPENDENT,
        NUM_BASIC_TYPES
    };

    BasicType basicType;

    static const AtomicType *UniformBool, *VaryingBool;
    static const AtomicType *UniformInt8, *VaryingInt8;
    static const AtomicType *UniformInt16, *VaryingInt16;
    static const AtomicType *UniformInt32, *VaryingInt32;
    static const AtomicType *UniformUInt8, *VaryingUInt8;
    static const AtomicType *UniformUInt16, *VaryingUInt16;
    static const AtomicType *UniformUInt32, *VaryingUInt32;
    static const AtomicType *UniformFloat16, *VaryingFloat16;
    static const AtomicType *UniformFloat, *VaryingFloat;
    static const AtomicType *UniformInt64, *VaryingInt64;
    static const AtomicType *UniformUInt64, *VaryingUInt64;
    static const AtomicType *UniformDouble, *VaryingDouble;
    static const AtomicType *Dependent;
    static const AtomicType *Void;

    /** The VaryingInt1 type is needed as a workaround for the absence of an
     * ISPC type (for some mask widths, e.g., i32) representing i1, which is
     * required for certain LLVM intrinsics like masked.load. In my opinion,
     * this is another consequence of the unclear bool/mask types in
     * ISPC. There is no intention of supporting in general, so only fixes
     * to ensure functionality for two specific intrinsics are made. */
    static const AtomicType *VaryingInt1;

  private:
    AtomicType(BasicType basicType, Variability v, bool isConst);

    // Copy constructor to allow cloning without enumerating all fields
    AtomicType(const AtomicType &other);

    AtomicType *create() const override;
    const AtomicType *createWithBasicType(BasicType newBasicType) const;
};

/** @brief Type representing a template typename type.

    'TemplateTypeParmType' should be resolved during template instantiation.
 */
class TemplateTypeParmType : public Type {
  public:
    TemplateTypeParmType(std::string, Variability v, bool ic, SourcePos pos);

    bool IsCompleteType() const override;

    const Type *GetAsSOAType(int width) const override;
    const Type *ResolveDependence(TemplateInstantiation &templInst) const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    std::string GetName() const;

  private:
    const std::string name;

    TemplateTypeParmType(const TemplateTypeParmType &other);

    TemplateTypeParmType *create() const override;
};

/** @brief Type implementation for enumerated types
 *
 *  Note that ISPC enum assumes 32 bit int as underlying type.
 */
class EnumType : public Type {
  public:
    /** Constructor for anonymous enumerated types */
    EnumType(SourcePos pos);
    /** Constructor for named enumerated types */
    EnumType(const char *name, SourcePos pos);

    bool IsIntType() const override;
    bool IsUnsignedType() const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    /** Returns the name of the enum type.  (e.g. struct Foo -> "Foo".) */
    const std::string &GetEnumName() const { return name; }

    /** Provides the enumerators defined in the enum definition. */
    void SetEnumerators(const std::vector<Symbol *> &enumerators);
    /** Returns the total number of enuemrators in this enum type. */
    int GetEnumeratorCount() const;
    /** Returns the symbol for the given enumerator number. */
    const Symbol *GetEnumerator(int i) const;

  private:
    const std::string name;
    std::vector<Symbol *> enumerators;

    EnumType(const std::string &name, Variability v, bool ic, SourcePos pos, const std::vector<Symbol *> &enumerators);
    EnumType(const EnumType &other);

    EnumType *create() const override;
};

/** @brief Type implementation for pointers to other types

    Pointers can have two additional properties beyond their variability
    and the type of object that they are pointing to.  Both of these
    properties are used for internal bookkeeping and aren't directly
    accessible from the language.

    - Slice: pointers that point to data with SOA layout have this
      property--it indicates that the pointer has two components, where the
      first (major) component is a regular pointer that points to an
      instance of the soa<> type being indexed, and where the second
      (minor) component is an integer that indicates which of the soa
      slices in that instance the pointer points to.

    - Frozen: only slice pointers may have this property--it indicates that
      any further indexing calculations should only be applied to the major
      pointer, and the value of the minor offset should be left unchanged.
      Pointers to lvalues from structure member access have the frozen
      property; see discussion in comments in the StructMemberExpr class.
 */

class PointerType : public Type {
  public:
    enum Property : unsigned int {
        NONE = 0,
        SLICE = 1,
        FROZEN = 2,
    };

    PointerType(const Type *t, Variability v, bool isConst, unsigned int prop = 0,
                AddressSpace as = AddressSpace::ispc_default);

    const Type *GetBaseType() const override;
    const PointerType *ResolveDependence(TemplateInstantiation &templInst) const override;
    const PointerType *ResolveUnboundVariability(Variability v) const override;
    const PointerType *GetAsConstType() const override;
    const PointerType *GetAsNonConstType() const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    /** Helper method to return a uniform pointer to the given type. */
    static PointerType *GetUniform(const Type *t, bool isSlice = false);
    /** Helper method to return a varying pointer to the given type. */
    static PointerType *GetVarying(const Type *t);

    /** Returns true if the given type is a void * type. */
    static bool IsVoidPointer(const Type *t);

    bool IsSlice() const { return property; }
    bool IsFrozenSlice() const { return property & FROZEN; }
    AddressSpace GetAddressSpace() const { return addrSpace; }
    const PointerType *GetAsSlice() const;
    const PointerType *GetAsNonSlice() const;
    const PointerType *GetAsFrozenSlice() const;

    const PointerType *GetWithAddrSpace(AddressSpace as) const;

    static PointerType *Void;

  private:
    unsigned int property;
    // TODO!: outline baseType and related methods from PointerType,
    // RefereceType, ArrayType and VectorType to TypeWithBaseType class
    const Type *baseType;
    // TODO!: storing address space in Type doesn't seem to be a correct
    AddressSpace addrSpace;

    PointerType(const PointerType &other);

    PointerType *create() const override;

    const PointerType *createWithBaseType(const Type *newBaseType) const;
    const PointerType *createWithAddressSpace(AddressSpace newAddrSpace) const;
    const PointerType *createWithBaseTypeAndVariability(const Type *newBaseType, Variability newVariability) const;
    const PointerType *createWithProperty(unsigned int newProperty) const;
    const PointerType *createWithConstAndProperty(bool newIsConst, unsigned int newProperty) const;
};

/** @brief Abstract base class for types that represent collections of
    other types.

    This is a common base class that StructTypes, ArrayTypes, and
    VectorTypes all inherit from.
*/
class CollectionType : public Type {
  public:
    /** Returns the total number of elements in the collection. */
    virtual int GetElementCount() const = 0;

    /** Returns the type of the element given by index.  (The value of
        index must be between 0 and GetElementCount()-1.
     */
    virtual const Type *GetElementType(int index) const = 0;

  protected:
    CollectionType(TypeId id, Variability v = Variability(Variability::Unbound), bool is_const = false,
                   SourcePos p = SourcePos(), unsigned int align = 0)
        : Type(id, v, is_const, p, align) {}
};

/** @brief Abstract base class for types that represent sequences

    SequentialType is an abstract base class that adds interface routines
    for types that represent linear sequences of other types (i.e., arrays
    and vectors).
 */
class SequentialType : public CollectionType {
  public:
    /** Number of elements in the vector/array or a symbol representing it */
    struct ElementCount {
        int fixedCount;
        Symbol *symbolCount;
        ElementCount(int count) : fixedCount(count), symbolCount(nullptr) {}
        ElementCount(Symbol *sym) : fixedCount(0), symbolCount(sym) {}
        ElementCount(int count, Symbol *sym) : fixedCount(count), symbolCount(sym) {}
    };

    /** Returns the Type of the elements that the sequence stores; for
        SequentialTypes, all elements have the same type . */
    virtual const Type *GetElementType() const { return base; }

    /** SequentialType provides an implementation of this CollectionType
        method, just passing the query on to the GetElementType(void)
        implementation, since all of the elements of a SequentialType have
        the same type.
     */
    const Type *GetElementType(int index) const override { return GetElementType(); }

    /* Returns true if the number of elements in the collection is dependent on a template parameter. */
    virtual bool IsCountDependent() const { return elementCount.symbolCount != nullptr; }

    int GetElementCount() const override { return elementCount.fixedCount; }

    const Type *GetBaseType() const override { return base; }

    bool IsCompleteType() const override;
    const SequentialType *GetAsVaryingType() const override;
    const SequentialType *GetAsUniformType() const override;
    const SequentialType *GetAsUnboundVariabilityType() const override;
    const SequentialType *GetAsSOAType(int width) const override;
    const SequentialType *ResolveUnboundVariability(Variability v) const override;

    const SequentialType *GetAsUnsignedType() const override;
    const SequentialType *GetAsSignedType() const override;
    const SequentialType *GetAsConstType() const override;
    const SequentialType *GetAsNonConstType() const override;

  protected:
    /** Base type that the SequentialType holds elements of */
    const Type *base;

    /** Number of elements in the SequentialType */
    ElementCount elementCount;

    SequentialType(TypeId id, const Type *b, ElementCount ec, Variability v = Variability(Variability::Unbound),
                   bool is_const = false, SourcePos pos = SourcePos(), unsigned int align = 0);

    const SequentialType *createWithBaseType(const Type *newBaseType) const;
    const SequentialType *createWithElementCount(ElementCount newElementCount) const;
    const SequentialType *createWithBaseTypeAndElementCount(const Type *newBaseType,
                                                            ElementCount newElementCount) const;

    /** Resolves the total number of elements in the collection in template instantiation. */
    virtual int ResolveElementCount(TemplateInstantiation &templInst) const = 0;
};

/** @brief One-dimensional array type.

    ArrayType represents a one-dimensional array of instances of some other
    type.  (Multi-dimensional arrays are represented by ArrayTypes that in
    turn hold ArrayTypes as their child types.)
*/
class ArrayType : public SequentialType {
  public:
    /** An ArrayType is created by providing the type of the elements that
        it stores, and the SOA width to use in laying out the array in
        memory.

        @param elementType  Type of the array elements
        @param numElements  Total number of elements in the array.  This
                            parameter may be zero, in which case this is an
                            "unsized" array type.  (Arrays of specific size
                            can be converted to unsized arrays to be passed
                            to functions that take array parameters, for
                            example).
     */
    ArrayType(const Type *elementType, int numElements);
    /** An ArrayType can be created with a symbolic number of elements.
        This constructor initializes an ArrayType with a specified element type
        and a symbolic representation of the number of elements. It is used when
        the size of the array is determined through template non-type parameters.

        @param elementType The type of the elements stored in the array.
        @param num A pointer to a Symbol representing the number of elements.
     */
    ArrayType(const Type *elementType, Symbol *num);
    /** This constructor initializes an ArrayType using an ElementCount structure,
        which can represent either a fixed count or a symbolic count of elements.

        @param elementType The type of the elements stored in the array.
        @param elCount An ElementCount structure representing the number of elements.
    */
    ArrayType(const Type *elementType, ElementCount elCount);

    bool IsCompleteType() const override;
    const Type *GetBaseType() const override;

    const ArrayType *ResolveDependence(TemplateInstantiation &templInst) const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;
    llvm::ArrayType *LLVMType(llvm::LLVMContext *ctx) const override;

    /** This method returns the total number of elements in the array,
        including all dimensions if this is a multidimensional array. */
    int TotalElementCount() const;

    /* Checks if the array is unsized.
       An array is considered unsized if its size is not explicitly set
       as compile-time constant i.e. `fixedCount` is 0 and `symbolCount`
       is nullptr. Unsized arrays are typically used as functions parameters.
    */
    bool IsUnsized() const;
    /** Returns a new array of the same child type, but with the given
        length. */
    const ArrayType *GetSizedArray(int length) const;

    /** If the given type is a (possibly multi-dimensional) array type and
        the initializer expression is an expression list, set the size of
        any array dimensions that are unsized according to the number of
        elements in the corresponding sectoin of the initializer
        expression.
     */
    static const Type *SizeUnsizedArrays(const Type *type, Expr *initExpr);

  private:
    /** Resolves the total number of elements in the array in template instantiation. */
    virtual int ResolveElementCount(TemplateInstantiation &templInst) const override;

    ArrayType(const ArrayType &other);

    ArrayType *create() const override;
};

/** @brief A (short) vector of atomic types.

    VectorType is used to represent a fixed-size array of elements of an
    AtomicType.  Vectors are similar to arrays in that they support
    indexing of the elements, but have two key differences.  First, all
    arithmetic and logical operations that are value for the element type
    can be performed on corresponding VectorTypes (as long as the two
    VectorTypes have the same size). Second, VectorTypes of uniform
    elements are laid out in memory aligned to the target's vector size;
    this allows them to be packed 'horizontally' into vector registers.
 */
class VectorType : public SequentialType {
  public:
    VectorType(const Type *base, int size);
    VectorType(const Type *base, Symbol *num);
    VectorType(const Type *base, ElementCount elCount);

    bool IsBoolType() const override;
    bool IsFloatType() const override;
    bool IsIntType() const override;
    bool IsUnsignedType() const override;
    bool IsSignedType() const override;

    const VectorType *ResolveDependence(TemplateInstantiation &templInst) const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMStorageType(llvm::LLVMContext *ctx) const override;
    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    std::string GetCountString() const;

  private:
    /** Resolves the total number of elements in the vector in template instantiation. */
    virtual int ResolveElementCount(TemplateInstantiation &templInst) const override;

    VectorType(const VectorType &other);

    VectorType *create() const override;

  public:
    /** Returns the number of elements stored in memory for the vector.
        For uniform vectors, this is rounded up so that the number of
        elements evenly divides the target's native vector width. */
    int getVectorMemoryCount() const;
};

/** @brief Representation of a structure holding a number of members.
 */
class StructType : public CollectionType {
  public:
    StructType(const std::string &name, const llvm::SmallVector<const Type *, 8> &elts,
               const llvm::SmallVector<std::string, 8> &eltNames, const llvm::SmallVector<SourcePos, 8> &eltPositions,
               bool isConst, Variability variability, SourcePos pos, unsigned int alignment = 0);

    bool IsCompleteType() const override;

    const StructType *GetAsSOAType(int width) const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    /** Returns the type of the i'th structure element.  The value of \c i must
        be between 0 and NumElements()-1. */
    const Type *GetElementType(int i) const override;

    /** Returns the total number of elements in the structure. */
    int GetElementCount() const override { return int(elementTypes.size()); }

    bool IsDefined() const;
    bool IsAnonymousType() const;

    const StructType *GetAsNamed(const std::string &name) const;

    /** Returns the type of the structure element with the given name (if any).
        Returns nullptr if there is no such named element. */
    const Type *GetElementType(const std::string &name) const;

    /** Return the raw element type without "finilizing" varibility and constness of the element. */
    const Type *GetRawElementType(int i) const;

    /** Returns which structure element number (starting from zero) that
        has the given name.  If there is no such element, return -1. */
    int GetElementNumber(const std::string &name) const;

    /** Returns the name of the i'th element of the structure. */
    const std::string &GetElementName(int i) const { return elementNames[i]; }

    const SourcePos &GetElementPosition(int i) const { return elementPositions[i]; }

    /** Returns the name of the structure type.  (e.g. struct Foo -> "Foo".) */
    const std::string &GetStructName() const { return name; }
    const std::string GetCStructName() const;

    void RegisterInStructTypeMap();

  private:
    static bool checkIfCanBeSOA(const StructType *st);

    /*const*/ std::string name;
    /** The types of the struct elements.  Note that we store these with
        uniform/varying exactly as they were declared in the source file.
        (In other words, even if this struct has a varying qualifier and
        thus all of its members are going to be widened out to be varying,
        we still store any members that were declared as uniform as uniform
        types in the elementTypes array, converting them to varying as
        needed in the implementation.)  This is so that if we later need to
        make a uniform version of the struct, we've maintained the original
        information about the member types.
     */
    const llvm::SmallVector<const Type *, 8> elementTypes;
    const llvm::SmallVector<std::string, 8> elementNames;
    /** Source file position at which each structure element declaration
        appeared. */
    const llvm::SmallVector<SourcePos, 8> elementPositions;
    bool isAnonymous;

    mutable llvm::SmallVector<const Type *, 8> finalElementTypes;

    mutable const StructType *oppositeConstStructType;

    StructType(const StructType &other);

    StructType *create() const override;
    const StructType *createWithVariability(Variability newVariability) const override;
    const StructType *createWithConst(bool newIsConst) const override;
    const StructType *createWithAlignment(unsigned int newAlignment) const override;
};

/** Type implementation representing a struct name that has been declared
    but where the struct members haven't been defined (i.e. "struct Foo;").
    This class doesn't do much besides serve as a placeholder that other
    code can use to detect the presence of such as truct.
 */
class UndefinedStructType : public Type {
  public:
    UndefinedStructType(const std::string &name, const Variability variability, bool isConst, SourcePos pos,
                        unsigned int alignment = 0);

    bool IsCompleteType() const override;

    const UndefinedStructType *GetAsSOAType(int width) const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    /** Returns the name of the structure type.  (e.g. struct Foo -> "Foo".) */
    const std::string &GetStructName() const { return name; }

    void RegisterInStructTypeMap();

  private:
    const std::string name;

    UndefinedStructType(const UndefinedStructType &other);

    UndefinedStructType *create() const override;
    const UndefinedStructType *createWithVariability(Variability newVariability) const override;
    const UndefinedStructType *createWithConst(bool newIsConst) const override;
    const UndefinedStructType *createWithAlignment(unsigned int newAlignment) const override;
};

/** @brief Type representing a reference to another (non-reference) type.
 */
class ReferenceType : public Type {
  public:
    ReferenceType(const Type *targetType, AddressSpace as = AddressSpace::ispc_default);

    bool IsBoolType() const override;
    bool IsFloatType() const override;
    bool IsIntType() const override;
    bool IsUnsignedType() const override;
    bool IsSignedType() const override;

    const Type *GetBaseType() const override;
    const Type *GetReferenceTarget() const override;
    const ReferenceType *GetAsVaryingType() const override;
    const ReferenceType *GetAsUniformType() const override;
    const ReferenceType *GetAsUnboundVariabilityType() const override;
    const Type *GetAsSOAType(int width) const override;
    const ReferenceType *ResolveDependence(TemplateInstantiation &templInst) const override;
    const ReferenceType *ResolveUnboundVariability(Variability v) const override;

    const ReferenceType *GetAsConstType() const override;
    const ReferenceType *GetAsNonConstType() const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    AddressSpace GetAddressSpace() const { return addrSpace; }

    const ReferenceType *GetWithAddrSpace(AddressSpace as) const;

  private:
    const Type *targetType;
    AddressSpace addrSpace;

    ReferenceType(const ReferenceType &other);

    ReferenceType *create() const override;
    const ReferenceType *createWithBaseType(const Type *newBaseType) const;
    const ReferenceType *createWithAddressSpace(AddressSpace newAddrSpace) const;
};

/** @brief Type representing a function (return type + argument types)

    FunctionType encapsulates the information related to a function's type,
    including the return type and the types of the arguments.

    @todo This class has a fair number of methods inherited from Type that
    don't make sense here (e.g. IsUniformType(), GetBaseType(), LLVMType(), etc.
    Would be nice to refactor the inheritance hierarchy to move most of
    those interface methods to a sub-class of Type, which in turn all of
    the other Type implementations inherit from.
 */
class FunctionType : public Type {
  public:
    enum FunctionFlag : unsigned int {
        FUNC_NONE = 0,
        FUNC_TASK = 1 << 0,          // 0x0001
        FUNC_EXPORTED = 1 << 1,      // 0x0002
        FUNC_EXTERNAL_ONLY = 1 << 2, // 0x0004
        FUNC_EXTERN_C = 1 << 3,      // 0x0008
        FUNC_EXTERN_SYCL = 1 << 4,   // 0x0010
        FUNC_UNMASKED = 1 << 5,      // 0x0020
        FUNC_UNMANGLED = 1 << 6,     // 0x0040
        FUNC_VECTOR_CALL = 1 << 7,   // 0x0080
        FUNC_REG_CALL = 1 << 8,      // 0x0100
        FUNC_CDECL = 1 << 9,         // 0x0200
        FUNC_SAFE = 1 << 10          // 0x0400
    };

    FunctionType(const Type *returnType, const llvm::SmallVector<const Type *, 8> &argTypes, SourcePos pos);
    FunctionType(const Type *returnType, const llvm::SmallVector<const Type *, 8> &argTypes,
                 const llvm::SmallVector<std::string, 8> &argNames, const llvm::SmallVector<Expr *, 8> &argDefaults,
                 const llvm::SmallVector<SourcePos, 8> &argPos, int costOverride, unsigned int flags, SourcePos p);
    // Structure holding the mangling suffix and prefix for function
    struct FunctionMangledName {
        std::string prefix;
        std::string suffix;
    };

    const Type *GetBaseType() const override;
    const Type *GetAsVaryingType() const override;
    const Type *GetAsUniformType() const override;
    const Type *GetAsUnboundVariabilityType() const override;
    const Type *GetAsSOAType(int width) const override;
    const FunctionType *ResolveDependence(TemplateInstantiation &templInst) const override;
    const FunctionType *ResolveUnboundVariability(Variability v) const override;

    const Type *GetAsConstType() const override;
    const Type *GetAsNonConstType() const override;

    std::string GetString() const override;
    std::string Mangle() const override;
    std::string GetDeclaration(const std::string &name, DeclarationSyntax syntax) const override;

    llvm::Type *LLVMType(llvm::LLVMContext *ctx) const override;
    llvm::DIType *GetDIType(llvm::DIScope *scope) const override;

    std::string GetDeclarationForDispatch(const std::string &fname, DeclarationSyntax syntax) const;

    bool IsISPCKernel() const;
    bool IsISPCExternal() const;

    const Type *GetAsUnmaskedType() const;
    const Type *GetAsNonUnmaskedType() const;
    const Type *GetWithReturnType(const Type *t) const;

    const Type *GetReturnType() const { return returnType; }

    const std::string GetReturnTypeString() const;

    /** This method returns the FunctionMangledName depending on Function type.
        The \c appFunction parameter indicates whether the function is generated for
        internal ISPC call or for external call from application.*/
    FunctionMangledName GetFunctionMangledName(bool appFunction, TemplateArgs *templateArgs = nullptr) const;

    /** This method returns std::vector of LLVM types of function arguments.
        The \c disableMask parameter indicates whether the mask parameter should be
        included to the list of arguments types. */
    std::vector<llvm::Type *> LLVMFunctionArgTypes(llvm::LLVMContext *ctx, bool disableMask = false) const;

    /** This method returns the LLVM FunctionType that corresponds to this
        function type.  The \c disableMask parameter indicates whether the
        llvm::FunctionType should have the trailing mask parameter, if
        present, removed from the return function signature. */
    llvm::FunctionType *LLVMFunctionType(llvm::LLVMContext *ctx, bool disableMask = false) const;

    /** This method creates LLVM Function that corresponds to this function type with provided name.
        The \c disableMask parameter indicates whether the llvm::FunctionType should have the trailing mask parameter,
       if present, removed from the return function signature. */
    llvm::Function *CreateLLVMFunction(const std::string &name, llvm::LLVMContext *ctx, bool disableMask = false) const;

    /* This method returns appropriate llvm::CallingConv for the function*/
    unsigned int GetCallingConv() const;

    /* Get string representation of calling convention */
    const std::string GetNameForCallConv() const;

    int GetNumParameters() const { return (int)paramTypes.size(); }
    const Type *GetParameterType(int i) const;
    Expr *GetParameterDefault(int i) const;
    const SourcePos &GetParameterSourcePos(int i) const;
    const std::string &GetParameterName(int i) const;

    int GetCostOverride() const { return costOverride; }

    /** Return true if the function had a 'task' qualifier in the
        source program. */
    bool IsTask() const { return flags & FUNC_TASK; }

    /** Return true if the function had a 'export' qualifier in the
        source program. */
    bool IsExported() const { return flags & FUNC_EXPORTED; }

    /** This signals compiler to omit generation of ISPC function copy
        for function with export qualifier, i.e., ISPC generates only the
        external function for calling from C/C++ */
    bool IsExternalOnly() const { return flags & FUNC_EXTERNAL_ONLY; }

    /** Return true if the function was declared as an 'extern "C"'
        function in the source program. */
    bool IsExternC() const { return flags & FUNC_EXTERN_C; }

    /** Return true if the function was declared as an 'extern "SYCL"'
    function in the source program. */
    bool IsExternSYCL() const { return flags & FUNC_EXTERN_SYCL; }

    /** Indicates whether the function doesn't take an implicit mask
        parameter (and thus should start execution with an "all on"
        mask). */
    bool IsUnmasked() const { return flags & FUNC_UNMASKED; }

    /** Indicates whether the function name should be mangled. */
    bool IsUnmangled() const { return flags & FUNC_UNMANGLED; }

    /** Indicates whether the function has __vectorcall attribute. */
    bool IsVectorCall() const { return flags & FUNC_VECTOR_CALL; }

    /** Indicates whether the function has __regcall attribute. */
    bool IsRegCall() const { return flags & FUNC_REG_CALL; }

    /** Indicates whether the function has __cdecl attribute (default C calling
        convention). It is useful when --vectorcall is provided globally to not
        call some function with vectorcall. */
    bool IsCdecl() const { return flags & FUNC_CDECL; }

    /** Indicates whether this function has been declared to be safe to run
        with an all-off mask. */
    bool IsSafe() const { return flags & FUNC_SAFE; }

  private:
    std::string mangleTemplateArgs(TemplateArgs *templateArgs) const;

    const Type *returnType;

    // The following four vectors should all have the same length (which is
    // in turn the length returned by GetNumParameters()).
    llvm::SmallVector<const Type *, 8> paramTypes;
    const llvm::SmallVector<std::string, 8> paramNames;
    /** Default values of the function's arguments.  For arguments without
        default values provided, nullptr is stored. */
    llvm::SmallVector<Expr *, 8> paramDefaults;
    /** The names provided (if any) with the function arguments in the
        function's signature.  These should only be used for error messages
        and the like and so not affect testing function types for equality,
        etc. */
    const llvm::SmallVector<SourcePos, 8> paramPositions;

    /** If non-negative, this provides a user-supplied override to the cost
        function estimate for the function. */
    int costOverride;

    unsigned int flags;

    mutable const FunctionType *asMaskedType, *asUnmaskedType;

    FunctionType(const FunctionType &other);

    FunctionType *create() const override;
    const FunctionType *createWithSignature(const Type *newReturnType,
                                            const llvm::SmallVector<const Type *, 8> &newParamTypes) const;
    const FunctionType *createWithFlags(unsigned int newFlags) const;
};

/* Efficient dynamic casting of Types.  First, we specify a default
   template function that returns nullptr, indicating a failed cast, for
   arbitrary types. */
template <typename T> inline const T *CastType(const Type *type) { return nullptr; }

/* Now we have template specializaitons for the Types implemented in this
   file.  Each one checks the Type::typeId member and then performs the
   corresponding static cast if it's safe as per the typeId.
 */
template <> inline const AtomicType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == ATOMIC_TYPE)
        return (const AtomicType *)type;
    else
        return nullptr;
}

template <> inline const TemplateTypeParmType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == TEMPLATE_TYPE_PARM_TYPE)
        return (const TemplateTypeParmType *)type;
    else
        return nullptr;
}

template <> inline const EnumType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == ENUM_TYPE)
        return (const EnumType *)type;
    else
        return nullptr;
}

template <> inline const PointerType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == POINTER_TYPE)
        return (const PointerType *)type;
    else
        return nullptr;
}

template <> inline const ArrayType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == ARRAY_TYPE)
        return (const ArrayType *)type;
    else
        return nullptr;
}

template <> inline const VectorType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == VECTOR_TYPE)
        return (const VectorType *)type;
    else
        return nullptr;
}

template <> inline const SequentialType *CastType(const Type *type) {
    // Note that this function must be updated if other sequential type
    // implementations are added.
    if (type != nullptr && (type->typeId == ARRAY_TYPE || type->typeId == VECTOR_TYPE))
        return (const SequentialType *)type;
    else
        return nullptr;
}

template <> inline const CollectionType *CastType(const Type *type) {
    // Similarly a new collection type implementation requires updating
    // this function.
    if (type != nullptr && (type->typeId == ARRAY_TYPE || type->typeId == VECTOR_TYPE || type->typeId == STRUCT_TYPE))
        return (const CollectionType *)type;
    else
        return nullptr;
}

template <> inline const StructType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == STRUCT_TYPE)
        return (const StructType *)type;
    else
        return nullptr;
}

template <> inline const UndefinedStructType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == UNDEFINED_STRUCT_TYPE)
        return (const UndefinedStructType *)type;
    else
        return nullptr;
}

template <> inline const ReferenceType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == REFERENCE_TYPE)
        return (const ReferenceType *)type;
    else
        return nullptr;
}

template <> inline const FunctionType *CastType(const Type *type) {
    if (type != nullptr && type->typeId == FUNCTION_TYPE)
        return (const FunctionType *)type;
    else
        return nullptr;
}

inline bool IsReferenceType(const Type *t) { return CastType<ReferenceType>(t) != nullptr; }

} // namespace ispc
