﻿/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/

#include "wallpaper.h"
#include <QDesktopServices>
#include <QFileInfo>
#include <QProcess>
#include <QFileSystemWatcher>
#include <QtConcurrent/QtConcurrent>

Wallpaper::Wallpaper()
    : mFirstLoad(true)
{
    pluginName = tr("Background");
    pluginType = PERSONALIZED;
    picInfoList.clear();
    QtConcurrent::run([=]{
        preInitialize();
    });
}

Wallpaper::~Wallpaper()
{
    if (wallpaperUi) {
        wallpaperUi->deleteLater();
        wallpaperUi = nullptr;
    }
}

QString Wallpaper::plugini18nName() {
    return pluginName;
}

int Wallpaper::pluginTypes() {
    return pluginType;
}

void Wallpaper::preInitialize() {
    wallpaperInterface = new QDBusInterface("org.ukui.ukcc.session",
                                           "/Wallpaper",
                                           "org.ukui.ukcc.session.Wallpaper",
                                           QDBusConnection::sessionBus(), this);
    if (!wallpaperInterface->isValid()) {
        qCritical() << "org.ukui.ukcc.session.Wallpaper DBus error:" << wallpaperInterface->lastError();
        return;
    }
    QStringList previewPictures = wallpaperInterface->property("previewWallpapers").toStringList();
    QStringList sourcePictures  = wallpaperInterface->property("sourceWallpapers").toStringList();
    if (previewPictures.size() != sourcePictures.size()) {
        qWarning() << "picturePathList.size():" << previewPictures.size() << " != sourcePathList.size():" << sourcePictures.size();
        return;
    }
    picInfoList.clear();
    for (int i = 0; i < previewPictures.size(); ++i) {
        PictureInfo* pictureInfo = new PictureInfo();
        if (pictureInfo == nullptr)
            continue;
        pictureInfo->pixmap.load(previewPictures.at(i));
        pictureInfo->fileName = sourcePictures.at(i);
        picInfoList.append(pictureInfo);
    }
}

QWidget *Wallpaper::pluginUi() {
    if (mFirstLoad) {
        mFirstLoad = false;
        wallpaperUi        = new WallpaperUi;
        initContent();
    } else {
        wallpaperUi->resize(wallpaperUi->size() - QSize(1, 1));
        wallpaperUi->resize(wallpaperUi->size() + QSize(1, 1));
    }
    return wallpaperUi;
}

const QString Wallpaper::name() const {

    return QStringLiteral("Wallpaper");
}

bool Wallpaper::isShowOnHomePage() const
{
    return true;
}

QIcon Wallpaper::icon() const
{
    return QIcon::fromTheme("folder-pictures-symbolic");
}

bool Wallpaper::isEnable() const
{
    return true;
}

void Wallpaper::dataChanged(const QString &key)
{
    if (toChangeKey != key) {
        if (key == "displayType") {
            initType();
        } else if (key == "wallpaper") {
            initMode();
        }
    }
    toChangeKey = "";
}

void Wallpaper::initContent()
{
    initModes();
    initTypes();
    initMode();
    initType();
    initPictures();
    connectUiSignals();
}

void Wallpaper::initModes()
{
    QStringList modes = {tr("picture"), tr("color")};
    QStringList modeData = {"picture", "color"};
    wallpaperUi->setWallpaperModes(modes, modeData);
}

void Wallpaper::initTypes()
{
    QStringList types    = {tr("scaled"), tr("wallpaper"), tr("centered"), tr("stretched"), tr("zoom"), tr("spanned")};
    QStringList typeData = {"scaled", "wallpaper", "centered", "stretched", "zoom", "spanned"};
    wallpaperUi->setDisplayTypes(types, typeData);
}

void Wallpaper::initMode()
{
    QString mode = wallpaperInterface->property("wallpaperMode").toString();
    if (mode == "color") {
        wallpaperUi->setWallpaperMode(tr("color"), wallpaperInterface->property("wallpaper").toString());
        wallpaperUi->showColors();
    } else if (mode == "picture") {
        wallpaperUi->setWallpaperMode(tr("picture"), wallpaperInterface->property("wallpaper").toString());
        wallpaperUi->showPictures();
    }
}

void Wallpaper::initType()
{
    QString type = wallpaperInterface->property("displayType").toString();
    if (type == "scaled") {
        wallpaperUi->setDisplayType(tr("scaled"));
    } else if (type == "wallpaper") {
        wallpaperUi->setDisplayType(tr("wallpaper"));
    } else if (type == "centered") {
        wallpaperUi->setDisplayType(tr("centered"));
    } else if (type == "stretched") {
        wallpaperUi->setDisplayType(tr("stretched"));
    } else if (type == "zoom") {
        wallpaperUi->setDisplayType(tr("zoom"));
    } else if (type == "spanned") {
        wallpaperUi->setDisplayType(tr("spanned"));
    }
}

void Wallpaper::connectUiSignals()
{
    QDBusConnection::sessionBus().connect("org.ukui.ukcc.session",
                                          "/Wallpaper",
                                          "org.ukui.ukcc.session.Wallpaper",
                                          "changed",
                                          this,
                                          SLOT(dataChanged(QString)));
    connect(wallpaperUi, &WallpaperUi::wallpaperChanged, this, [=](QString mode, QString data){
        toChangeKey = "wallpaper";
        wallpaperInterface->call("setWallpaper", mode, data);
        UkccCommon::buriedSettings(name(), QString("Background"), QString("select"), mode + "/" + data);
    });
    connect(wallpaperUi, &WallpaperUi::typeChanged, this, [=](QString type){
        toChangeKey = "displayType";
        wallpaperInterface->call("setDisplayType", type);
        UkccCommon::buriedSettings(name(), QString("Type"), QString("select"), type);
    });
    connect(wallpaperUi, &WallpaperUi::localButtonClicked, this, [=](){
        showLocalWpDialog();
        UkccCommon::buriedSettings(name(), QString("Local Pictures"), QString("clicked"));
    });
    connect(wallpaperUi, &WallpaperUi::onlineButtonClicked, this, [=](){
        QDesktopServices::openUrl(QUrl(QLatin1String("https://www.ubuntukylin.com/wallpaper.html")));
        UkccCommon::buriedSettings(name(), QString("Online Pictures"), QString("clicked"));
    });
    connect(wallpaperUi, &WallpaperUi::resetButtonClicked, this, [=](){
        wallpaperInterface->call("resetDefault");
        UkccCommon::buriedSettings(name(), QString("Reset To Default"), QString("clicked"));
    });
}

void Wallpaper::initPictures()
{
    wallpaperUi->setPicturesFile(picInfoList);
}

void Wallpaper::showLocalWpDialog()
{
    QStringList filters;
    filters<<tr("Wallpaper files(*.jpg *.jpeg *.bmp *.dib *.png *.jfif *.jpe *.gif *.tif *.tiff *.wdp)");
    QFileDialog fd(wallpaperUi);
    QList<QUrl> usb_list = fd.sidebarUrls();
    int sidebarNum = 8;// 最大添加U盘数，可以自己定义
    QString home_path = QDir::homePath().section("/", -1, -1);
    QString mnt = "/media/" + home_path + "/";
    QDir mntDir(mnt);
    mntDir.setFilter(QDir::Dirs | QDir::NoDotAndDotDot);
    QFileInfoList file_list = mntDir.entryInfoList();
    QList<QUrl> mntUrlList;
    for (int i = 0; i < sidebarNum && i < file_list.size(); ++i) {
        QFileInfo fi = file_list.at(i);
        mntUrlList << QUrl("file://" + fi.filePath());
    }

    QFileSystemWatcher m_fileSystemWatcher(&fd);
    m_fileSystemWatcher.addPath("/media/" + home_path + "/");
    connect(&m_fileSystemWatcher, &QFileSystemWatcher::directoryChanged, &fd,
            [=, &sidebarNum, &mntUrlList, &usb_list, &fd](const QString path) {
        QDir m_wmntDir(path);
        m_wmntDir.setFilter(QDir::Dirs | QDir::NoDotAndDotDot);
        QFileInfoList m_wfilist = m_wmntDir.entryInfoList();
        mntUrlList.clear();
        for (int i = 0; i < sidebarNum && i < m_wfilist.size(); ++i) {
            QFileInfo m_fi = m_wfilist.at(i);
            mntUrlList << QUrl("file://" + m_fi.filePath());
        }
        fd.setSidebarUrls(usb_list + mntUrlList);
        fd.update();
    });

    connect(&fd, &QFileDialog::finished, &fd, [=, &usb_list, &fd]() {
        fd.setSidebarUrls(usb_list);
    });

    fd.setDirectory(QString(const_cast<char *>(g_get_user_special_dir(G_USER_DIRECTORY_PICTURES))));
    fd.setAcceptMode(QFileDialog::AcceptOpen);
    fd.setViewMode(QFileDialog::List);
    fd.setNameFilters(filters);
    fd.setFileMode(QFileDialog::ExistingFile);
    fd.setWindowTitle(tr("select custom wallpaper file"));
    fd.setLabelText(QFileDialog::Accept, tr("Select"));
    fd.setLabelText(QFileDialog::LookIn, tr("Position: "));
    fd.setLabelText(QFileDialog::FileName, tr("FileName: "));
    fd.setLabelText(QFileDialog::FileType, tr("FileType: "));
    fd.setLabelText(QFileDialog::Reject, tr("Cancel"));

    fd.setSidebarUrls(usb_list + mntUrlList);

    if (fd.exec() != QDialog::Accepted) {
        return;
    }
    QString selectedfile;
    selectedfile = fd.selectedFiles().first();

    QStringList fileRes = selectedfile.split("/");

    QProcess process;
    QString program("cp");
    QStringList arguments;
    arguments << selectedfile ;
    arguments << "/tmp";
    process.start(program, arguments);
    QString bgfile = "/tmp/" + fileRes.at(fileRes.length() - 1);

    wallpaperInterface->call("setWallpaper", "picture", selectedfile);
}
