/*  
   Sequence Generator - seq-gen, version 1.3.4
   Copyright (c)1996-2017, Andrew Rambaut & Nick Grassly
   Institute of Evolutionary Biology, University of Edinburgh			
   All rights reserved.                          

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

     1. Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.

     2. Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.

     3. The names of its contributors may not be used to endorse or promote 
        products derived from this software without specific prior written 
        permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


   Any feedback is very welcome.
   http://tree.bio.ed.ac.uk/software/seqgen/
   email: a.rambaut@ed.ac.uk
*/
 
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include "eigen.h"
#include "aamodels.h"

char *aminoAcids="ARNDCQEGHILKMFPSTWYV";

int aaFreqSet;

int aaModel = AA_NONE;
double aaFreq[NUM_AA];
double aaAddFreq[NUM_AA];
double aaMatrix[MAX_RATE_CATS][SQNUM_AA];
double aaVector[NUM_AA];

double aaRelativeRate[NUM_AA_REL_RATES];
static double Qij[SQNUM_AA], Cijk[CUNUM_AA], Root[NUM_AA];

void SetupAAMatrix();
void SetRelativeRates(double *inRelativeRates);
void SetFrequencies(double *inFrequencies);
void CheckAAFrequencies();

/* JTT model for amino acid evolution */
/* D.T. Jones, W.R. Taylor, and J.M. Thornton */
/* The rapid generation of mutation data matrices from protein sequences */
/* CABIOS  vol. 8 no. 3 1992 pp. 275-282 */
static double jttRelativeRates[NUM_AA_REL_RATES] = {
	0.531678, 0.557967, 0.827445, 0.574478, 0.556725, 1.066681, 1.740159, 0.219970, 0.361684, 0.310007, 0.369437, 0.469395, 0.138293, 1.959599, 3.887095, 4.582565, 0.084329, 0.139492, 2.924161,
	0.451095, 0.154899, 1.019843, 3.021995, 0.318483, 1.359652, 3.210671, 0.239195, 0.372261, 6.529255, 0.431045, 0.065314, 0.710489, 1.001551, 0.650282, 1.257961, 0.235601, 0.171995,
	5.549530, 0.313311, 0.768834, 0.578115, 0.773313, 4.025778, 0.491003, 0.137289, 2.529517, 0.330720, 0.073481, 0.121804, 5.057964, 2.351311, 0.027700, 0.700693, 0.164525,
	0.105625, 0.521646, 7.766557, 1.272434, 1.032342, 0.115968, 0.061486, 0.282466, 0.190001, 0.032522, 0.127164, 0.589268, 0.425159, 0.057466, 0.453952, 0.315261,
	0.091304, 0.053907, 0.546389, 0.724998, 0.150559, 0.164593, 0.049009, 0.409202, 0.678335, 0.123653, 2.155331, 0.469823, 1.104181, 2.114852, 0.621323,
	3.417706, 0.231294, 5.684080, 0.078270, 0.709004, 2.966732, 0.456901, 0.045683, 1.608126, 0.548807, 0.523825, 0.172206, 0.254745, 0.179771,
	1.115632, 0.243768, 0.111773, 0.097485, 1.731684, 0.175084, 0.043829, 0.191994, 0.312449, 0.331584, 0.114381, 0.063452, 0.465271,
	0.201696, 0.053769, 0.069492, 0.269840, 0.130379, 0.050212, 0.208081, 1.874296, 0.316862, 0.544180, 0.052500, 0.470140,
	0.181788, 0.540571, 0.525096, 0.329660, 0.453428, 1.141961, 0.743458, 0.477355, 0.128193, 5.848400, 0.121827,
	2.335139, 0.202562, 4.831666, 0.777090, 0.098580, 0.405119, 2.553806, 0.134510, 0.303445, 9.533943,
	0.146481, 3.856906, 2.500294, 1.060504, 0.592511, 0.272514, 0.530324, 0.241094, 1.761439,
	0.624581, 0.024521, 0.216345, 0.474478, 0.965641, 0.089134, 0.087904, 0.124066,
	0.436181, 0.164215, 0.285564, 2.114728, 0.201334, 0.189870, 3.038533,
	0.148483, 0.943971, 0.138904, 0.537922, 5.484236, 0.593478,
	2.788406, 1.176961, 0.069965, 0.113850, 0.211561,
	4.777647, 0.310927, 0.628608, 0.408532,
	0.080556, 0.201094, 1.14398,
	0.747889, 0.239697,
	0.165473
};

static double jttFrequencies[NUM_AA] = {
	0.076862, 0.051057, 0.042546, 0.051269, 0.020279, 0.041061, 0.061820, 0.074714, 0.022983, 0.052569, 0.091111, 0.059498, 0.023414, 0.040530, 0.050532, 0.068225, 0.058518, 0.014336, 0.032303, 0.066374
};
	
/* WAG model of amino acid evolution */
/* Whelan, S. and Goldman, N. (2001) A general empirical model of protein */
/* evolution derived from multiple protein families using a maximum-likelihood */
/* approach. Mol. Biol. Evol. 18, 691699. */
static double wagRelativeRates[NUM_AA_REL_RATES] = {
	0.610810, 0.569079, 0.821500, 1.141050, 1.011980, 1.756410, 1.572160, 0.354813, 0.219023, 0.443935, 1.005440, 0.989475, 0.233492, 1.594890, 3.733380, 2.349220, 0.125227, 0.268987, 2.221870, 
	0.711690, 0.165074, 0.585809, 3.360330, 0.488649, 0.650469, 2.362040, 0.206722, 0.551450, 5.925170, 0.758446, 0.116821, 0.753467, 1.357640, 0.613776, 1.294610, 0.423612, 0.280336, 
	6.013660, 0.296524, 1.716740, 1.056790, 1.253910, 4.378930, 0.615636, 0.147156, 3.334390, 0.224747, 0.110793, 0.217538, 4.394450, 2.257930, 0.078463, 1.208560, 0.221176, 
	0.033379, 0.691268, 6.833400, 0.961142, 1.032910, 0.043523, 0.093930, 0.533362, 0.116813, 0.052004, 0.472601, 1.192810, 0.417372, 0.146348, 0.363243, 0.169417, 
	0.109261, 0.023920, 0.341086, 0.275403, 0.189890, 0.428414, 0.083649, 0.437393, 0.441300, 0.122303, 1.560590, 0.570186, 0.795736, 0.604634, 1.114570, 
	6.048790, 0.366510, 4.749460, 0.131046, 0.964886, 4.308310, 1.705070, 0.110744, 1.036370, 1.141210, 0.954144, 0.243615, 0.252457, 0.333890, 
	0.630832, 0.635025, 0.141320, 0.172579, 2.867580, 0.353912, 0.092310, 0.755791, 0.782467, 0.914814, 0.172682, 0.217549, 0.655045, 
	0.276379, 0.034151, 0.068651, 0.415992, 0.194220, 0.055288, 0.273149, 1.486700, 0.251477, 0.374321, 0.114187, 0.209108, 
	0.152215, 0.555096, 0.992083, 0.450867, 0.756080, 0.771387, 0.822459, 0.525511, 0.289998, 4.290350, 0.131869, 
	3.517820, 0.360574, 4.714220, 1.177640, 0.111502, 0.353443, 1.615050, 0.234326, 0.468951, 8.659740, 
	0.287583, 5.375250, 2.348200, 0.462018, 0.382421, 0.364222, 0.740259, 0.443205, 1.997370, 
	1.032220, 0.098843, 0.619503, 1.073780, 1.537920, 0.152232, 0.147411, 0.342012, 
	1.320870, 0.194864, 0.556353, 1.681970, 0.570369, 0.473810, 2.282020, 
	0.179896, 0.606814, 0.191467, 1.699780, 7.154480, 0.725096, 
	1.786490, 0.885349, 0.156619, 0.239607, 0.351250, 
	4.847130, 0.578784, 0.872519, 0.258861, 
	0.126678, 0.325490, 1.547670, 
	2.763540, 0.409817, 
	0.347826
};

static double wagFrequencies[NUM_AA] = {
	0.0866, 0.0440, 0.0391, 0.0570, 0.0193, 0.0367, 0.0581, 0.0833, 0.0244, 0.0485, 0.0862, 0.0620, 0.0195, 0.0384, 0.0458, 0.0695, 0.0610, 0.0144, 0.0353, 0.0709
};

/* LG model of amino acid evolution */
/* Le, S. and Gascuel, O. (2008) An improved Amino Acid Replacement Matrix */
/* Mol. Biol. Evol. 25(7):1307-1320. */
static double lgRelativeRates[NUM_AA_REL_RATES] = {
	0.425093, 0.276818, 0.395144, 2.489084, 0.969894, 1.038545, 2.066040, 0.358858, 0.149830, 0.395337, 0.536518, 1.124035, 0.253701, 1.177651, 4.727182, 2.139501, 0.180717, 0.218959, 2.547870,
	0.751878, 0.123954, 0.534551, 2.807908, 0.363970, 0.390192, 2.426601, 0.126991, 0.301848, 6.326067, 0.484133, 0.052722, 0.332533, 0.858151, 0.578987, 0.593607, 0.314440, 0.170887,
	5.076149, 0.528768, 1.695752, 0.541712, 1.437645, 4.509238, 0.191503, 0.068427, 2.145078, 0.371004, 0.089525, 0.161787, 4.008358, 2.000679, 0.045376, 0.612025, 0.083688,
	0.062556, 0.523386, 5.243870, 0.844926, 0.927114, 0.010690, 0.015076, 0.282959, 0.025548, 0.017416, 0.394456, 1.240275, 0.425860, 0.029890, 0.135107, 0.037967,
	0.084808, 0.003499, 0.569265, 0.640543, 0.320627, 0.594007, 0.013266, 0.893680, 1.105251, 0.075382, 2.784478, 1.143480, 0.670128, 1.165532, 1.959291,
	4.128591, 0.267959, 4.813505, 0.072854, 0.582457, 3.234294, 1.672569, 0.035855, 0.624294, 1.223828, 1.080136, 0.236199, 0.257336, 0.210332,
	0.348847, 0.423881, 0.044265, 0.069673, 1.807177, 0.173735, 0.018811, 0.419409, 0.611973, 0.604545, 0.077852, 0.120037, 0.245034,
	0.311484, 0.008705, 0.044261, 0.296636, 0.139538, 0.089586, 0.196961, 1.739990, 0.129836, 0.268491, 0.054679, 0.076701,
	0.108882, 0.366317, 0.697264, 0.442472, 0.682139, 0.508851, 0.990012, 0.584262, 0.597054, 5.306834, 0.119013,
	4.145067, 0.159069, 4.273607, 1.112727, 0.078281, 0.064105, 1.033739, 0.111660, 0.232523, 10.649107,
	0.137500, 6.312358, 2.592692, 0.249060, 0.182287, 0.302936, 0.619632, 0.299648, 1.702745,
	0.656604, 0.023918, 0.390322, 0.748683, 1.136863, 0.049906, 0.131932, 0.185202,
	1.798853, 0.099849, 0.346960, 2.020366, 0.696175, 0.481306, 1.898718,
	0.094464, 0.361819, 0.165001, 2.457121, 7.803902, 0.654683,
	1.338132, 0.571468, 0.095131, 0.089613, 0.296501,
	6.472279, 0.248862, 0.400547, 0.098369,
	0.140825, 0.245841, 2.188158,
	3.151815, 0.189510,
	0.249313
};

static double lgFrequencies[NUM_AA] = {
  0.079066, 0.055941, 0.041977, 0.053052, 0.012937, 0.040767, 0.071586, 0.057337, 0.022355, 0.062157, 0.099081, 0.064600, 0.022951, 0.042302, 0.044040, 0.061197, 0.053287, 0.012066, 0.034155, 0.069147
};

/* HIVB model of amino acid evolution */
/* Nickle DC, Heath L, Jensen MA, Gilbert PB, Mullins JI, Kosakovsky Pond SL */
/* HIV-Specific Probabilistic Models of Protein Evolution */
/* PLoS ONE 2(6): e503. */
static double hivbRelativeRates[NUM_AA_REL_RATES] = {
	0.307507, 0.005,    1.45504,  0.123758, 0.055113, 1.48135,  2.13536,  0.084761, 0.005,    0.215256, 0.005,    0.018664, 0.014127, 2.12217, 2.46633,  15.9183,  0.005,    0.005,    7.61428,
	0.295543, 0.005,    0.351721, 3.4215,   0.074922, 3.65345,  9.04044,  0.677289, 0.701427, 20.45,    2.51394,  0.005,    1.28355,  3.4791,  2.86868,  0.991338, 0.009918, 0.081245,
	17.6612,  0.086064, 0.672052, 0.079263, 0.323401, 7.64585,  0.680565, 0.005,    7.90443,  0.005,    0.005,    0.007396, 13.1447,  6.88667, 0.005,    1.76417,  0.026656,
	0.005,    0.005,    10.5872,  2.83806,  1.9169,   0.017679, 0.00876,  0.005,    0.005,    0.005,    0.034266, 0.52823,  0.274724, 0.005,   0.674653, 1.04793,
	0.005,    0.005,    0.897871, 0.240073, 0.005,    0.129777, 0.005,    0.005,    9.29815,  0.005,    4.69314,  0.739969, 2.63277,  7.57932, 0.420027,
	2.5602,   0.061914, 7.05545,  0.005,    1.49456,  6.54737,  0.303676, 0.005,    4.47211,  0.116311, 0.243589, 0.026656, 0.113033, 0.020915,
	3.92775,  0.11974,  0.006091, 0.005,    4.61482,  0.175789, 0.005,    0.012023, 0.005,    0.289774, 0.005,    0.079263, 1.02847,
	0.005,    0.005,    0.005,    0.521705, 0.005,    0.291561, 0.005,    4.38041,  0.369615, 1.21674,  0.005,    0.953155,
	0.103111, 1.74171,  0.005,    0.005,    0.145558, 2.45318,  0.382747, 0.711594, 0.069518, 18.6943,  0.005,
	5.95879,  0.322319, 11.2065,  3.39836,  0.041059, 1.21803,  8.61217,  0.005,    0.148168, 17.7389,
	0.0815,   5.31961,  8.52484,  2.07757,  0.927656, 0.043767, 0.748843, 0.111986, 1.41036,
	1.28246,  0.034266, 0.031386, 0.504111, 4.67142,  0.005,    0.005,    0.265829,
	0.188025, 0.005,    0.005,    4.94026,  0.089078, 0.005,    6.8532,
	0.005,    0.956472, 0.014127, 0.829343, 15.34,    0.723274,
	5.37762,  2.01417,  0.044451, 0.030438, 0.005,
	8.93107,  0.024873, 0.648024, 0.074922,
	0.005,    0.105652, 0.709226,
	1.28022,  0.005,
	0.041059
}; 
static double hivbFrequencies[NUM_AA] = {
  0.060490222, 0.066039665, 0.044127815, 0.042109048, 0.020075899, 0.053606488, 0.071567447, 0.072308239, 0.022293943, 0.069730629, 0.098851122, 0.056968211, 0.019768318, 0.028809447, 0.046025282, 0.05060433, 0.053636813, 0.033011601, 0.028350243, 0.061625237
};


/* Dayhoff, M.O., Schwartz, R.M., Orcutt, B.C. (1978)*/
/* A model of evolutionary change in proteins.*/
/* Dayhoff, M.O. (ed.) Atlas of Protein Sequence Structur., Vol5, Suppl. 3,*/
/* National Biomedical Research Foundation, Washington DC, pp. 345-352.*/
static double dayhoffRelativeRates[NUM_AA_REL_RATES] = {
	0.267828, 0.984474, 1.199805, 0.360016, 0.887753, 1.961167, 2.386111, 0.228116, 0.653416, 0.406431, 0.258635, 0.717840, 0.183641, 2.485920, 4.051870, 3.680365, 0.000000, 0.244139, 2.059564,
	0.327059, 0.000000, 0.232374, 2.439939, 0.000000, 0.087791, 2.383148, 0.632629, 0.154924, 4.610124, 0.896321, 0.136906, 1.028313, 1.531590, 0.265745, 2.001375, 0.078012, 0.240368,
	8.931515, 0.000000, 1.028509, 1.493409, 1.385352, 5.290024, 0.768024, 0.341113, 3.148371, 0.000000, 0.138503, 0.419244, 4.885892, 2.271697, 0.224968, 0.946940, 0.158067,
	0.00000, 1.348551, 11.388659, 1.240981, 0.868241, 0.239248, 0.000000, 0.716913, 0.000000, 0.000000, 0.133940, 0.956097, 0.660930, 0.000000, 0.000000, 0.178316,
	0.000000, 0.000000, 0.107278, 0.282729, 0.438074, 0.000000, 0.000000, 0.000000, 0.000000, 0.187550, 1.598356, 0.162366, 0.000000, 0.953164, 0.484678,
	7.086022, 0.281581, 6.011613, 0.180393, 0.730772, 1.519078, 1.127499, 0.000000, 1.526188, 0.561828, 0.525651, 0.000000, 0.000000, 0.346983,
	0.811907, 0.439469, 0.609526, 0.112880, 0.830078, 0.304803, 0.000000, 0.507003, 0.793999, 0.340156, 0.000000, 0.214717, 0.36725,
	0.106802, 0.000000, 0.071514, 0.267683, 0.170372, 0.153478, 0.347153, 2.322243, 0.306662, 0.000000, 0.000000, 0.538165,
	0.076981, 0.443504, 0.270475, 0.000000, 0.475927, 0.933709, 0.353643, 0.226333, 0.270564, 1.265400, 0.438715,
	2.556685, 0.460857, 3.332732, 1.951951, 0.119152, 0.247955, 1.900739, 0.000000, 0.374834, 8.810038,
	0.180629, 5.230115, 1.565160, 0.316258, 0.171432, 0.331090, 0.461776, 0.286572, 1.745156,
	2.411739, 0.000000, 0.335419, 0.954557, 1.350599, 0.000000, 0.132142, 0.10385,
	0.921860, 0.170205, 0.619951, 1.031534, 0.000000, 0.000000, 2.565955,
	0.110506, 0.459901, 0.136655, 0.762354, 6.952629, 0.123606,
	2.427202, 0.782857, 0.000000, 0.000000, 0.485026,
	5.436674, 0.740819, 0.336289, 0.303836,
	0.000000, 0.417839, 1.561997,
	0.608070, 0.000000,
	0.279379
};

static double dayhoffFrequencies[NUM_AA] = {
	0.087127, 0.040904, 0.040432, 0.046872, 0.033474, 0.038255, 0.049530, 0.088612, 0.033619, 0.036886, 0.085357, 0.080481, 0.014753, 0.039772, 0.050680, 0.069577, 0.058542, 0.010494, 0.029916, 0.064718
};

/* BLOSUM62 model of amino acid evolution */
/* Henikoff, S., and J. G. Henikoff. 1992. PNAS USA 89:10915-10919.*/
static double blosumRelativeRates[NUM_AA_REL_RATES] = {
	7.3579038969751e-01, 4.8539105546575e-01, 5.4316182089867e-01, 1.4599953104700e+00, 1.1997057046020e+00, 1.1709490427999e+00, 1.9558835749595e+00, 7.1624144499779e-01, 6.0589900368677e-01, 8.0001653051838e-01, 1.2952012667833e+00, 1.2537582666635e+00, 4.9296467974759e-01, 1.1732759009239e+00, 4.3250926870566e+00, 1.7291780194850e+00, 4.6583936772479e-01, 7.1820669758623e-01, 2.1877745220045e+00, 
	1.2974467051337e+00, 5.0096440855513e-01, 2.2782657420895e-01, 3.0208336100636e+00, 1.3605741904203e+00, 4.1876330851753e-01, 1.4561411663360e+00, 2.3203644514174e-01, 6.2271166969249e-01, 5.4111151414889e+00, 9.8369298745695e-01, 3.7164469320875e-01, 4.4813366171831e-01, 1.1227831042096e+00, 9.1466595456337e-01, 4.2638231012175e-01, 7.2051744121611e-01, 4.3838834377202e-01, 
	3.1801000482161e+00, 3.9735894989702e-01, 1.8392161469920e+00, 1.2404885086396e+00, 1.3558723444845e+00, 2.4145014342081e+00, 2.8301732627800e-01, 2.1188815961519e-01, 1.5931370434574e+00, 6.4844127878707e-01, 3.5486124922252e-01, 4.9488704370192e-01, 2.9041016564560e+00, 1.8981736345332e+00, 1.9148204624678e-01, 5.3822251903674e-01, 3.1285879799342e-01, 
	2.4083661480204e-01, 1.1909457033960e+00, 3.7616252083685e+00, 7.9847324896839e-01, 7.7814266402188e-01, 4.1855573246161e-01, 2.1813157759360e-01, 1.0324479249521e+00, 2.2262189795786e-01, 2.8173069420651e-01, 7.3062827299842e-01, 1.5827541420653e+00, 9.3418750943056e-01, 1.4534504627853e-01, 2.6142220896504e-01, 2.5812928941763e-01, 
	3.2980150463028e-01, 1.4074889181440e-01, 4.1820319228376e-01, 3.5405810983129e-01, 7.7489402279418e-01, 8.3184264014158e-01, 2.8507880090648e-01, 7.6768882347954e-01, 4.4133747118660e-01, 3.5600849876863e-01, 1.1971884150942e+00, 1.1198313585160e+00, 5.2766441887169e-01, 4.7023773369610e-01, 1.1163524786062e+00, 
	5.5289191779282e+00, 6.0984630538281e-01, 2.4353411311401e+00, 2.3620245120365e-01, 5.8073709318144e-01, 3.9452776745146e+00, 2.4948960771127e+00, 1.4435695975031e-01, 8.5857057567418e-01, 1.9348709245965e+00, 1.2774802945956e+00, 7.5865380864172e-01, 9.5898974285014e-01, 5.3078579012486e-01, 
	4.2357999217628e-01, 1.6268910569817e+00, 1.8684804693170e-01, 3.7262517508685e-01, 2.8024271516787e+00, 5.5541539747043e-01, 2.9140908416530e-01, 9.2656393484598e-01, 1.7698932389373e+00, 1.0710972360073e+00, 4.0763564893830e-01, 5.9671930034577e-01, 5.2425384633796e-01, 
	5.3985912495418e-01, 1.8929629237636e-01, 2.1772115923623e-01, 7.5204244030271e-01, 4.5943617357855e-01, 3.6816646445253e-01, 5.0408659952683e-01, 1.5093262532236e+00, 6.4143601140497e-01, 5.0835892463812e-01, 3.0805573703500e-01, 2.5334079019018e-01, 
	2.5271844788492e-01, 3.4807220979697e-01, 1.0225070358890e+00, 9.8431152535870e-01, 7.1453370392764e-01, 5.2700733915060e-01, 1.1170297629105e+00, 5.8540709022472e-01, 3.0124860078016e-01, 4.2189539693890e+00, 2.0155597175031e-01, 
	3.8909637733035e+00, 4.0619358664202e-01, 3.3647977631042e+00, 1.5173593259539e+00, 3.8835540920564e-01, 3.5754441245967e-01, 1.1790911972601e+00, 3.4198578754023e-01, 6.7461709322842e-01, 8.3118394054582e+00, 
	4.4557027426059e-01, 6.0305593795716e+00, 2.0648397032375e+00, 3.7455568747097e-01, 3.5296918452729e-01, 9.1525985769421e-01, 6.9147463459998e-01, 8.1124585632307e-01, 2.2314056889131e+00, 
	1.0730611843319e+00, 2.6692475051102e-01, 1.0473834507215e+00, 1.7521659178195e+00, 1.3038752007987e+00, 3.3224304063396e-01, 7.1799348690032e-01, 4.9813847530407e-01, 
	1.7738551688305e+00, 4.5412362510273e-01, 9.1872341574605e-01, 1.4885480537218e+00, 8.8810109815193e-01, 9.5168216224591e-01, 2.5758507553153e+00, 
	2.3359790962888e-01, 5.4002764482413e-01, 4.8820611879305e-01, 2.0743248934965e+00, 6.7472604308008e+00, 8.3811961017754e-01, 
	1.1691295777157e+00, 1.0054516831488e+00, 2.5221483002727e-01, 3.6940531935451e-01, 4.9690841067567e-01, 
	5.1515562922704e+00, 3.8792562209837e-01, 7.9675152076106e-01, 5.6192545744165e-01, 
	5.1312812689059e-01, 8.0101024319939e-01, 2.2530740511763e+00, 
	4.0544190065580e+00, 2.6650873142646e-01, 
	1.0000000000000e+00
};

static double blosumFrequencies[NUM_AA] = {
	0.074, 0.052, 0.045, 0.054, 0.025, 0.034, 0.054, 0.074, 0.026, 0.068, 0.099, 0.058, 0.025, 0.047, 0.039, 0.057, 0.051, 0.013, 0.032, 0.073
};

/* mtRev model - complete sequence data of mtDNA from 24 vertebrate species */
/* Adachi, J., and Hasegawa, M. 1996. J. Mol. Evol. 42:459-468. */
static double mtrevRelativeRates[NUM_AA_REL_RATES] = {
	1.2199217606346e+01, 1.4182139942122e+01, 9.2985091873208e+00, 3.1542792981957e+01, 1.0025852846688e+00, 5.1418866803338e+00, 6.3531246495131e+01, 7.3137132861715e+00, 5.0782382656186e+01, 1.3399741808481e+01, 4.4021672780560e+00, 7.4673480520104e+01, 3.3513021631978e+00, 2.8582502221773e+01, 2.0413623195312e+02, 2.5301305153906e+02, 1.0000000000000e+00, 3.4084158197615e+00, 1.0266468401249e+02, 
	6.9661274444534e+00, 1.0000000000000e+00, 5.4384584796568e+01, 1.1631134513343e+02, 1.0000000000000e+00, 1.2122831341194e+01, 8.6961067087353e+01, 1.0000000000000e+00, 8.1976829394538e+00, 7.4423215395318e+01, 1.0000000000000e+00, 2.4659158338099e+00, 1.2439947713615e+01, 3.1791814866372e+00, 1.0935327216119e+00, 1.1550775790126e+01, 1.0000000000000e+00, 4.0211417480338e+00, 
	4.1809325468160e+02, 3.1020979842967e+01, 9.1349622725361e+01, 3.3185663516310e+01, 2.8052324651124e+01, 2.6112087577885e+02, 1.4261453863336e+01, 7.9775653461977e+00, 3.2036829276162e+02, 3.4424354918739e+01, 7.9996445145608e+00, 3.8586541461044e+01, 2.6020426225852e+02, 1.2550758780474e+02, 5.6207759736659e+00, 1.0071406219571e+02, 1.0000000000000e+00, 
	1.0000000000000e+00, 2.9097352675564e+01, 3.0713149855302e+02, 2.9877072751897e+01, 5.9995408885817e+01, 2.2827096245105e+00, 1.0000000000000e+00, 1.2183938185384e+00, 1.0000000000000e+00, 2.6221929413096e+00, 7.0708004204733e+00, 3.6327934317139e+01, 1.4743408713748e+01, 1.0453246057102e+01, 1.1165627147496e+01, 1.0000000000000e+00, 
	3.9599394038972e+01, 1.0000000000000e+00, 1.6163581056674e+01, 7.4467985406234e+01, 3.3018175376623e+01, 1.3500725995091e+01, 1.0000000000000e+00, 3.2504095376923e+00, 3.7264767083096e+01, 1.6454136037822e+01, 1.4581783243113e+02, 9.4720031458442e+01, 1.7684087896962e+01, 1.3409157685926e+02, 1.0000000000000e+00, 
	1.6503249008836e+02, 3.5530760735494e+00, 3.0652523140859e+02, 4.3905393139325e+00, 2.0895470525345e+01, 2.4504076430724e+02, 2.4931300477797e+01, 1.0059428264289e+01, 7.2256314165467e+01, 2.8480937892158e+01, 4.9962974409828e+01, 1.0000000000000e+00, 2.0430790980529e+01, 9.9986289000676e+00, 
	1.4884496769963e+01, 2.5853576435567e+01, 1.7418201388328e+00, 1.0000000000000e+00, 1.6519126809071e+02, 1.0000000000000e+00, 1.4067850525292e+00, 6.7547121641947e+00, 2.8794794140840e+01, 7.8001372062558e+00, 1.0000000000000e+00, 6.9067239183061e+00, 1.1127702362585e+01, 
	1.0000000000000e+00, 3.1466649021550e+00, 1.2699794194865e+00, 1.1962111069278e+01, 1.0000000000000e+00, 1.0000000000000e+00, 1.0000000000000e+00, 6.6277950574411e+01, 5.8800079133028e+00, 5.7494182626674e+00, 1.6887657206208e+00, 1.3320553471351e+00, 
	6.4536986087271e+00, 6.0472584534958e+00, 6.7197196398961e+01, 6.2977633277779e+00, 2.5347805183364e+01, 3.2089868698728e+01, 4.0766987134407e+01, 2.3570850628539e+01, 3.7286635325194e+00, 3.5270764890474e+02, 1.0000000000000e+00, 
	1.7320653206333e+02, 1.0298655619743e+01, 2.7262244199514e+02, 4.4561065036310e+01, 1.0856482766156e+01, 2.5107659603898e+01, 1.9391167162525e+02, 1.0000000000000e+00, 1.3161329199391e+01, 6.4365086389428e+02, 
	7.8314019154706e+00, 2.8290920517725e+02, 1.1371735519833e+02, 2.1105885757279e+01, 3.8741359395934e+01, 6.6524559321657e+01, 1.7071378554833e+01, 2.3234516108847e+01, 4.8247261078055e+01, 
	4.8092094826036e+01, 3.3887559483420e+00, 2.6368577564199e+01, 5.5679895711418e+01, 7.1750284708933e+01, 1.2631893872825e+01, 2.6932728996777e+01, 1.0000000000000e+00, 
	4.7798798034572e+01, 9.9165053447429e+00, 5.8505442466161e+01, 2.7798190504760e+02, 1.1427000119701e+01, 2.1029990530586e+01, 2.0397078683768e+02, 
	9.1089574817139e+00, 3.3835737720574e+01, 1.7815549567056e+01, 4.1272404968214e+00, 2.4504156395152e+02, 3.3435675442163e+00, 
	8.9421193040709e+01, 6.7485067008375e+01, 2.2161693733113e+00, 8.5338209390745e+00, 4.3342126659660e+00, 
	3.1432036618746e+02, 2.0305343047059e+01, 3.4167877957799e+01, 1.0000000000000e+00, 
	5.2559565123081e+00, 2.0382362288681e+01, 1.0765527137500e+02, 
	1.3814733274637e+01, 2.8259139240676e+00, 
	1.0000000000000e+00
};

static double mtrevFrequencies[NUM_AA] = {
	0.072, 0.019, 0.039, 0.019, 0.006, 0.025, 0.024, 0.056, 0.028, 0.088, 0.168, 0.023, 0.054, 0.061, 0.054, 0.072, 0.086, 0.029, 0.033, 0.043
};

/* CPREV 45 model of amino acid evolution */
/* Adachi, J., P.J. Waddell, W. Martin, and M. Hasegawa. 2000. JME 50:348-358 */
static double cprevRelativeRates[NUM_AA_REL_RATES] = {
	-105, -227, -175, -669, -157, -499, -665, -66, -145, -197, -236, -185, -68, -490, -2440, -1340, -14, -56, -968, 
	-357, -43, -823, -1745, -152, -243, -715, -136, -203, -4482, -125, -53, -87, -385, -314, -230, -323, -92, 
	-4435, -538, -768, -1055, -653, -1405, -168, -113, -2430, -61, -97, -173, -2085, -1393, -40, -754, -83, 
	-10, -400, -3691, -431, -331, -10, -10, -412, -47, -22, -170, -590, -266, -18, -281, -75, 
	-10, -10, -303, -441, -280, -396, -48, -159, -726, -285, -2331, -576, -435, -1466, -592, 
	-3122, -133, -1269, -92, -286, -3313, -202, -10, -323, -396, -241, -53, -391, -54, 
	-379, -162, -148, -82, -2629, -113, -145, -185, -568, -369, -63, -142, -200, 
	-19, -40, -20, -263, -21, -25, -28, -691, -92, -82, -10, -91, 
	-29, -66, -305, -10, -127, -152, -303, -32, -69, -1971, -25, 
	-1745, -345, -1772, -454, -117, -216, -1040, -42, -89, -4797, 
	-218, -1351, -1268, -219, -516, -156, -159, -189, -865, 
	-193, -72, -302, -868, -918, -10, -247, -249, 
	-327, -100, -93, -645, -86, -215, -475, 
	-43, -487, -148, -468, -2370, -317, 
	-1202, -260, -49, -97, -122, 
	-2151, -73, -522, -167, 
	-29, -71, -760, 
	-346, -10, 
	-119
};

static double cprevFrequencies[NUM_AA] = {
	0.076, 0.062, 0.041, 0.037, 0.009, 0.038, 0.049, 0.084, 0.025, 0.081, 0.101, 0.050, 0.022, 0.051, 0.043, 0.062, 0.054, 0.018, 0.031, 0.066
};

/*
MtArt model inserted in following

7 Oct 2011 -- Lars Jermiin (lars.jermiin@csiro.au)
*/

/* MtArt: A New Model of Amino Acid Replacement for Arthropoda */
/* Abascal, F., Posada, D., and Zardoya, R. 2007. Mol. Biol. Evol. 24:1-5. */
static double mtartRelativeRates[NUM_AA_REL_RATES] = {
	0.19, 0.19, 0.59, 253.51, 0.19, 0.19, 199.84, 0.19, 25.69, 3.70, 0.19, 120.64, 13.10, 49.33, 672.97, 243.93, 0.19, 1.18, 339.91,
	0.19, 4.28, 35.51, 153.97, 0.19, 0.19, 41.30, 1.81, 1.77, 208.60, 5.18, 4.73, 0.19, 2.66, 0.19, 0.19, 3.88, 0.19,
	500.16, 98.18, 261.79, 183.04, 120.53, 179.54, 21.33, 12.65, 467.34, 78.81, 19.72, 16.54, 398.43, 165.89, 7.73, 251.17, 22.60,
	10.61, 0.19, 861.83, 12.46, 0.19, 6.60, 1.17, 1.68, 0.19, 0.19, 0.19, 44.35, 0.19, 0.19, 0.19, 0.19,
	0.19, 0.19, 80.53, 12.37, 63.00, 78.71, 0.19, 312.30, 184.06, 0.19, 664.16, 182.80, 21.61, 71.99, 350.40,
	261.55, 2.64, 313.50, 10.54, 16.28, 349.28, 67.33, 0.19, 39.30, 52.36, 43.68, 6.68, 86.67, 0.19,
	43.86, 15.25, 6.81, 1.72, 106.31, 0.19, 0.19, 7.88, 31.50, 43.40, 10.99, 7.74, 13.63,
	0.19, 2.74, 1.36, 0.19, 55.71, 0.83, 0.19, 226.03, 0.19, 1.88, 8.65, 2.56,
	0.19, 5.54, 0.19, 0.19, 13.78, 0.79, 10.62, 18.59, 0.19, 191.43, 0.19,
	514.54, 3.46, 514.78, 117.91, 0.19, 7.15, 203.74, 0.19, 12.33, 1854.52,
	3.78, 885.50, 262.57, 12.17, 8.17, 47.75, 21.13, 19.76, 84.72,
	105.61, 10.73, 16.82, 144.22, 69.54, 15.97, 117.09, 26.06,
	321.61, 5.26, 111.74, 288.57, 70.68, 70.92, 281.29,
	14.62, 36.05, 13.54, 53.67, 791.56, 51.90,
	86.50, 46.83, 0.19, 18.39, 31.70,
	660.39, 2.38, 30.45, 60.59,
	0.19, 45.98, 544.14,
	37.72, 0.19,
	1.59
};

static double mtartFrequencies[NUM_AA] = {
	0.054116, 0.018227, 0.039903, 0.020160, 0.009709, 0.018781, 0.024289, 0.068183, 0.024518, 0.092639, 0.148658, 0.021718, 0.061453, 0.088668, 0.041826, 0.091030, 0.049194, 0.029786, 0.039443, 0.057701
};

/*************************************/
void SetAAModel(int theAAModel)
{	
	aaModel = theAAModel;
	
/*
"AA_MTART" inserted in position 6 of the following switch
 
7 Oct 2011 -- Lars Jermiin (lars.jermiin@csiro.au)
*/
	switch (aaModel) {
		case AA_JTT: SetRelativeRates(jttRelativeRates); break;
		case AA_WAG: SetRelativeRates(wagRelativeRates); break;
		case AA_DAYHOFF78: SetRelativeRates(dayhoffRelativeRates); break;
		case AA_BLOSUM62: SetRelativeRates(blosumRelativeRates); break;
		case AA_MTREV24: SetRelativeRates(mtrevRelativeRates); break;
		case AA_CPREV45: SetRelativeRates(cprevRelativeRates); break;
		case AA_MTART: SetRelativeRates(mtartRelativeRates); break;
		case AA_LG: SetRelativeRates(lgRelativeRates); break;
		case AA_HIVB: SetRelativeRates(hivbRelativeRates); break;
		case AA_GENERAL: 
			// relative rates set by user
		break;
	}

/*
"AA_MTART" inserted in position 6 of the following switch
 
7 Oct 2011 -- Lars Jermiin (lars.jermiin@csiro.au)
*/
		
	if (!aaFreqSet) {
		switch (aaModel) {
			case AA_JTT: SetFrequencies(jttFrequencies); break;
			case AA_WAG: SetFrequencies(wagFrequencies); break;
			case AA_DAYHOFF78: SetFrequencies(dayhoffFrequencies); break;
			case AA_BLOSUM62: SetFrequencies(blosumFrequencies); break;
			case AA_MTREV24: SetFrequencies(mtrevFrequencies); break;
			case AA_CPREV45: SetFrequencies(cprevFrequencies); break;
			case AA_MTART: SetFrequencies(mtartFrequencies); break;
			case AA_LG: SetFrequencies(lgFrequencies); break;
			case AA_HIVB: SetFrequencies(hivbFrequencies); break;
			case AA_GENERAL: 
				// frequencies set by user
			break;
		}
	} else {
		// User set or equal frequencies
		CheckAAFrequencies();
	}
	
	SetupAAMatrix();
}
		
void SetRelativeRates(double *inRelativeRate) 
{
	int i;
	for (i=0; i<NUM_AA_REL_RATES; i++) {
		aaRelativeRate[i] = inRelativeRate[i];
	}
}

void SetFrequencies(double *inFrequencies)
{
	int i;
	for (i=0; i<NUM_AA; i++) {
		aaFreq[i] = inFrequencies[i];
	}
}

void SetupAAMatrix()
{
	int i,j,k;
	double mr;
	double sum;
	double U[SQNUM_AA], V[SQNUM_AA], T1[SQNUM_AA], T2[SQNUM_AA];

	k=0;
	for (i=0; i<NUM_AA-1; i++) {
		for (j=i+1; j<NUM_AA; j++) {
			Qij[i*NUM_AA+j] = Qij[j*NUM_AA+i] = aaRelativeRate[k++];
		}
	}
	
	for (i=0; i<NUM_AA; i++) {
		for (j=0; j<NUM_AA; j++) { 
			Qij[i*NUM_AA+j] *= aaFreq[j];
		}
	}

	mr=0;		
	for (i=0; i<NUM_AA; i++) {
		sum = 0;
		Qij[i*NUM_AA+i]=0; 
		for (j=0; j<NUM_AA; j++) { 
			sum += Qij[i*NUM_AA+j];
		}
		Qij[i*NUM_AA+i] = -sum; 
		mr += aaFreq[i] * sum;
	}
	
	abyx(1.0/mr, Qij, SQNUM_AA);

	if ((k=eigen(1, Qij, NUM_AA, Root, T1, U, V, T2))!=0) {
		fprintf(stderr, "\ncomplex roots in SetupAAMatrix");
		exit(0);
	}
	xtoy (U, V, SQNUM_AA);
	matinv (V, NUM_AA, NUM_AA, T1);
	for (i=0; i<NUM_AA; i++) {
   		for (j=0; j<NUM_AA; j++) {
   			for (k=0; k<NUM_AA; k++) {
   				Cijk[i*SQNUM_AA+j*NUM_AA+k] = U[i*NUM_AA+k]*V[k*NUM_AA+j];
   			}
   		}
   	}
}

/**
 * Ensures that frequencies are not smaller than MINFREQ and
 * that two frequencies differ by at least 2*MINFDIFF.
 * This avoids potential problems later when eigenvalues
 * are computed.
 */
void CheckAAFrequencies()
{
	int i, j;
	double diff;
	
	// required frequency difference
	double MINFDIFF = 1.0E-10;

	// lower limit on frequency
	double MINFREQ = 1.0E-10;

	int maxi = 0;
	double sum = 0.0;
	double maxfreq = 0.0;
	for (i = 0; i < NUM_AA; i++) {
		if (aaFreq[i] < MINFREQ) aaFreq[i] = MINFREQ;
		if (aaFreq[i] > maxfreq) {
			maxfreq = aaFreq[i];
			maxi = i;
		}
		sum += aaFreq[i];
	}
	
	diff = 1.0 - sum;
	aaFreq[maxi] += diff;

	for (i = 0; i < NUM_AA - 1; i++) {
		for (j = i+1; j < NUM_AA; j++) {
			if (aaFreq[i] == aaFreq[j]) {
				aaFreq[i] += MINFDIFF;
				aaFreq[j] -= MINFDIFF;
			}
		}
	}
}

void SetAAMatrix(double *matrix, double len)
{	
	int i,j,k;
	double expt[NUM_AA];
	double *P;
	
/* P(t)ij = SUM Cijk * exp{Root*t}
*/
	P=matrix;
	if (len<1e-6) { 
		for (i=0; i<NUM_AA; i++) {
			for (j=0; j<NUM_AA; j++) {
				if (i==j)
					*P=1.0;
				else 	
					*P=0.0;
				P++;
			}
		}
		return; 
	}
	
	for (k=1; k<NUM_AA; k++) {
		expt[k]=exp(len*Root[k]);
	}
	for (i=0; i<NUM_AA; i++) {
		for (j=0; j<NUM_AA; j++) {
			(*P)=Cijk[i*SQNUM_AA+j*NUM_AA];
			for (k=1; k<NUM_AA; k++) {
				(*P)+=Cijk[i*SQNUM_AA+j*NUM_AA+k]*expt[k];
			}
			P++;
		}
	}
	
	P=matrix;
	for (i=0; i<NUM_AA; i++) {
		P++;
		for (j=1; j<NUM_AA; j++) {
			*P += *(P - 1);
			P++;
		}
	}
}

void SetAAVector(double *vector, short state, double len)
{	
	int i,j,k;
	double expt[NUM_AA];
	double *P;

	P=vector;
	if (len<1e-6) { 
		for (i=0; i<NUM_AA; i++) {
			if (i==state)
				*P=1.0;
			else 	
				*P=0.0;
			P++;
		}
		return; 
	}
	for (k=1; k<NUM_AA; k++) {
		expt[k]=exp(len*Root[k]);
	}
	
	for (j=0; j<NUM_AA; j++) {
		(*P)=Cijk[state*SQNUM_AA+j*NUM_AA];
		for (k=1; k<NUM_AA; k++) {
			(*P)+=Cijk[state*SQNUM_AA+j*NUM_AA+k]*expt[k];
		}
		P++;
	}
	
	P=vector;
	P++;
	for (j=1; j<NUM_AA; j++) {
		*P += *(P - 1);
		P++;
	}
}


